<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Campaign;
use Carbon\Carbon;
use App\Models\Donation;
use Illuminate\Http\Request;

class StatisticController extends Controller
{
    public function statistics()
    {
        $pageTitle = "Donation Statistics";
        return view('admin.donation.statistics', compact('pageTitle'));
    }

    public function statisticsChart(Request $request)
    {
        $diffInDays = Carbon::parse($request->start_date)->diffInDays(Carbon::parse($request->end_date));

        $groupBy = $diffInDays > 30 ? 'months' : 'days';
        $format = $diffInDays > 30 ? '%M-%Y'  : '%d-%M-%Y';

        if ($groupBy == 'days') {
            $dates = $this->getAllDates($request->start_date, $request->end_date);
        } else {
            $dates = $this->getAllMonths($request->start_date, $request->end_date);
        }

        $getDonation   = Donation::paid()
            ->whereDate('created_at', '>=', $request->start_date)
            ->whereDate('created_at', '<=', $request->end_date)
            ->selectRaw('SUM(donation) AS amount')
            ->selectRaw("DATE_FORMAT(created_at, '{$format}') as created_on")
            ->latest()
            ->groupBy('created_on')
            ->get();

        $data = [];

        foreach ($dates as $date) {
            $data[] = [
                'created_on' => $date,
                'credits' => getAmount($getDonation->where('created_on', $date)->first()?->amount ?? 0)
            ];
        }

        $data = collect($data);

        // Monthly Deposit & Withdraw Report Graph
        $report['created_on']   = $data->pluck('created_on');
        $report['data']     = [
            [
                'name' => 'Successful Donation',
                'data' => $data->pluck('credits')
            ]
        ];

        return response()->json($report);
    }

    private function getAllDates($startDate, $endDate)
    {
        $dates = [];
        $currentDate = new \DateTime($startDate);
        $endDate = new \DateTime($endDate);

        while ($currentDate <= $endDate) {
            $dates[] = $currentDate->format('d-F-Y');
            $currentDate->modify('+1 day');
        }

        return $dates;
    }

    private function  getAllMonths($startDate, $endDate)
    {
        if ($endDate > now()) {
            $endDate = now()->format('Y-m-d');
        }

        $startDate = new \DateTime($startDate);
        $endDate = new \DateTime($endDate);

        $months = [];

        while ($startDate <= $endDate) {
            $months[] = $startDate->format('F-Y');
            $startDate->modify('+1 month');
        }

        return $months;
    }

    public function index()
    {
        $pageTitle = "All Donations";
        $donations = Donation::where('status', Status::DONATION_PAID)->orderBy('id', 'DESC')
                ->searchable(['fullname', 'mobile', 'donation', 'email', 'country', 'campaign:title', 'deposit.gateway:name', 'deposit:trx'])
                ->filter(['anonymous'])
                ->dateFilter()->with('campaign', 'deposit', 'deposit.gateway')
                ->filter(['status'])->whereHas('campaign')
                ->paginate(getPaginate());
        return view('admin.donation.index', compact('pageTitle', 'donations'));
    }

    public function campaignWiseDonations($campaignId)
    {
        $donations = Donation::where('status', Status::DONATION_PAID)->where('campaign_id', $campaignId)->orderBy('id', 'DESC')->searchable(['fullname', 'mobile', 'donation', 'campaign:title'])->with('campaign', 'deposit', 'deposit.gateway')->whereHas('campaign')->paginate(getPaginate());
        $pageTitle = Campaign::where('id', $campaignId)->first()->title;
        return view('admin.donation.index', compact('pageTitle', 'donations'));
    }
}
