<?php

namespace App\Http\Controllers;

use App\Constants\Status;
use App\Models\Campaign;
use App\Models\Category;
use App\Models\Comment;
use App\Models\Donation;
use App\Models\Page;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class CampaignController extends Controller
{
    public function index()
    {
        $pageTitle = 'Campaigns';
        $campaigns = Campaign::running()->boundary()->active()->with(['user.organization', 'donations', 'category']);

        if (request()->category) {
            $campaigns->whereHas('category', function ($q) {
                $q->where('slug', request()->category);
            });
        }

        if (request()->slug) {
            $category  = Category::where('slug', request()->slug)->active()->firstOrFail();
            $campaigns = $campaigns->where('category_id', $category->id);
        }

        $campaigns  = $campaigns->orderBy('id', 'DESC')->paginate(getPaginate());

        $categories = Category::active()->hasCampaigns()->orderBy('id', 'DESC')->get();

        $sections    = Page::where('tempname', activeTemplate())->where('slug', 'campaign')->first();
        $seoContents = @$sections->seo_content;
        $seoImage    = @$seoContents->image ? getImage(getFilePath('seo') . '/' . @$seoContents->image, getFileSize('seo')) : null;

        return view('Template::campaign.index', compact('pageTitle', 'campaigns', 'categories', 'sections', 'seoContents', 'seoImage'));
    }

    public function filterCampaign(Request $request)
    {
        $type  = '';
        $query = Campaign::active()->running()->boundary()->with(['user.organization', 'donations', 'category']);

        if ($request->search) {
            $query =   $query->searchable(['title']);
        }

        if ($request->category_id) {
            $query = $query->where('category_id', $request->category_id);
            $type  = Category::where('id', $request->category_id)->first()->name;
        }

        if ($request->criteria == 'urgent') {
            $query =   $query->whereDate('deadline', '>', now())
                ->whereDate('deadline', '<=', Carbon::now()->addDays(7));
            $type = 'Urgent';
        } else if ($request->criteria == 'featured') {
            $query = $query->where('featured', Status::YES);
            $type  = 'Featured';
        } else if ($request->criteria == 'top') {
            $campaigns =  $query->select('campaigns.*')->addSelect([
                'donations_sum' => Donation::select(DB::raw("SUM(donation)"))
                    ->whereColumn('campaign_id', 'campaigns.id')
                    ->paid()
            ])->orderBy('donations_sum', 'DESC')->paginate(getPaginate());
            $type = 'Top';
            return view('Template::campaign.campaign', compact('campaigns', 'type'));
        }

        if ($request->date) {
            $query =  $query->whereDate('created_at', '=', $request->date);
        }

        $campaigns =  $query->orderBy('id', 'DESC')->paginate(getPaginate());

        return view('Template::campaign.campaign', compact('campaigns', 'type'));
    }

    public function details($slug)
    {
        $user      = auth()->user();
        $isAdmin   = auth()->guard('admin')->check();
        $query     = Campaign::where('slug', $slug)->boundary();

        if (!$isAdmin) {
            if ($user) {
                $query->where(function ($query) use ($user) {
                    $query->where('user_id', $user->id);
                });
            } else {
                $query->running();
            }

            if (!$query->exists()) {
                $query = Campaign::where('slug', $slug)->running();
            }
        }

        $campaign     = $query->with('user', 'category', 'comments.user')->first();

        if (!$campaign) {
            $notify[] =  ['error', 'Campaign not found!'];
            return back()->withNotify($notify);
        }

        $pageTitle =  $campaign->title;

        //start-seo//
        $seoContents = $campaign->seo_content;
        $seoImage    = @$seoContents->image ? frontendImage('campaign', $seoContents->image, getFileSize('seo'), true) : null;

        $myFavorites = @auth()->user()?->favorites->pluck('campaign_id')->toArray();
        return view('Template::campaign.details', compact('pageTitle', 'campaign', 'myFavorites', 'seoContents', 'seoImage'));
    }

    public function comment(Request $request)
    {
        $request->validate([
            'campaign' => 'required|numeric',
            'review'   => 'required|string',
            'rating'   => 'required|integer|between:1,5',
        ]);

        $user = auth()->user();
        if (!$user) {
            $notify[] = ['error', 'Login is required for Campaign review'];
            return back()->withNotify($notify);
        }

        $campaign = Campaign::where('id', $request->campaign)->firstOrFail();

        if ($campaign->user_id == $user->id) {
            $notify[] = ['error', 'You cannot review your own campaign'];
            return back()->withNotify($notify);
        }

        $existingReview = Comment::where('campaign_id', $campaign->id)
            ->where('user_id', $user->id)
            ->first();

        if ($existingReview) {
            $notify[] = ['error', 'Already submitted'];
            return back()->withNotify($notify);
        }

        $comment              = new Comment();
        $comment->user_id     = $user->id;
        $comment->campaign_id = $request->campaign;
        $comment->rating      = $request->rating;
        $comment->review      = $request->review;
        $comment->save();

        $notify[] = ['success', 'Review added successfully, please wait for admin approval'];
        return back()->withNotify($notify);
    }

    public function widget($id)
    {
        header("Access-Control-Allow-Origin: *");
        $campaign = Campaign::running()->boundary()->with(['donations', 'user.organization'])->find($id);
        if (!$campaign) {
            return;
        }
        $donation    = $campaign->donations->where('status', Status::DONATION_PAID)->sum('donation');
        $percent     = percent($donation, $campaign);
        $progressBar = progressPercent($percent);
        $data = [
            'profile_name'     => $campaign->user->enable_org  ? $campaign->user->organization->name   :  $campaign->user->fullname,
            'user_image'       => getImage(getFilePath('campaign') . '/' . $campaign->image, getFileSize('campaign')),
            'title'            => $campaign->title,
            'progress_percent' => showAmount($progressBar, currencyFormat: false),
            'description'      => strLimit($campaign->description, 95)
        ];
        return response()->json($data);
    }

    public function thanksMessage($slug)
    {
        $pageTitle = 'Thank You';

        $user = auth()->user();

        if ($user) {
            $pageTitle = 'Thank You ' . @$user->fullname;
        }

        $campaign  = Campaign::where('slug', $slug)->first();
        $user      = $campaign->user;
        return view('Template::campaign.thanks_message', compact('pageTitle', 'campaign', 'user'));
    }

    public function loadMoreComments(Request $request)
    {
        try {
            // Validate inputs
            $request->validate([
                'campaign_id' => 'required|integer|exists:campaigns,id',
                'page' => 'nullable|integer|min:1',
            ]);

            $campaignId = $request->input('campaign_id');
            $page = $request->input('page', 1);

            // Fetch comments with eager loading
            $comments = Comment::with('user')
                ->where('campaign_id', $campaignId)
                ->where('status', Status::PUBLISHED)
                ->orderBy('id', 'DESC')
                ->paginate(5, ['*'], 'page', $page);

            $html = '';
            foreach ($comments as $comment) {
                $html .= view('Template::partials.review_item', compact('comment'))->render();
            }

            // Return JSON response
            return response()->json([
                'html' => $html,
                'hasMore' => $comments->hasMorePages(),
            ]);
        } catch (\Exception $e) {
            // Handle errors
            return response()->json([
                'error' => 'Unable to load more comments. Please try again.',
            ], 500);
        }
    }

    public function donationCampaign(Request $request)
    {
        $campaigns = Campaign::running()
            ->boundary()->with([
                'donations' => function ($query) {
                    $query->where('status', Status::DONATION_PAID);
                }
            ])
            ->withSum(
                [
                    'donations' => function ($query) {
                        $query->where('status', Status::DONATION_PAID);
                    }
                ],
                'donation',
            )
            ->orderBy('id', 'DESC');

        if ($request->search) {
            $campaigns = $campaigns->where('title', 'like', "%$request->search%");
        }

        $campaigns = $campaigns->paginate(15);
        $hasMore = $campaigns->hasPages();

        $formattedData = [];
        foreach ($campaigns as $campaign) {
            $formattedData[] = [
                'id' => $campaign->id,
                'title' => strLimit($campaign->title, 45),
                'goal' => showAmount($campaign->goal),
                'raised' => showAmount($campaign->donations_sum_donation)
            ];
        }

        return response()->json([
            'has_more' => $hasMore,
            'items' => $formattedData
        ]);
    }
}
