<?php

namespace App\Http\Controllers\Gateway;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Models\AdminNotification;
use App\Models\Deposit;
use App\Models\GatewayCurrency;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;

class PaymentController extends Controller {
    public function deposit() {
        if (!session()->has('DONATION')) {
            $notify[] = ['success', "Contributions are currently welcomed. Please consider supporting a campaign."];
            return to_route('campaign.index')->withNotify($notify);
        }
        $donation        = session()->get('DONATION');
        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->with('method')->orderby('name')->get();

        $pageTitle = 'Payment Methods';
        return view('Template::user.payment.deposit', compact('gatewayCurrency', 'donation', 'pageTitle'));
    }

    public function depositInsert(Request $request) {
        if (!session()->has('DONATION')) {
            $notify[] = ['success', "Contribution is currently welcomed. Please consider supporting a campaign."];
            return to_route('campaign.index')->withNotify($notify);
        }

        $request->validate([
            'amount'   => 'required|numeric|gt:0',
            'gateway'  => 'required',
            'currency' => 'required',
        ]);

        $gate = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->where('method_code', $request->gateway)->where('currency', $request->currency)->first();
        if (!$gate) {
            $notify[] = ['error', 'Invalid gateway'];
            return back()->withNotify($notify);
        }

        if ($gate->min_amount > $request->amount || $gate->max_amount < $request->amount) {
            $notify[] = ['error', 'Please follow payment limit'];
            return back()->withNotify($notify);
        }

        $donation = session()->get('DONATION');

        if (!$donation->campaign_id) {
            $notify[] = ['error', 'Please select a valid campaign!'];
            return back()->withNotify($notify);
        }

        $charge      = $gate->fixed_charge + ($request->amount * $gate->percent_charge / 100);
        $payable     = $request->amount + $charge;
        $finalAmount = $payable * $gate->rate;

        $data                  = new Deposit();
        $data->user_id         = $donation->campaign->user_id; //campaign_creator
        $data->donation_id     = $donation->id;
        $data->method_code     = $gate->method_code;
        $data->method_currency = strtoupper($gate->currency);
        $data->amount          = $request->amount;
        $data->charge          = $charge;
        $data->rate            = $gate->rate;
        $data->final_amount    = $finalAmount;
        $data->btc_amount      = 0;
        $data->btc_wallet      = "";
        $data->trx             = getTrx();
        $data->success_url     = urlPath('campaign.thanks', $donation->campaign->slug);
        $data->failed_url      = urlPath('campaign.index');
        $data->save();
        session()->put('Track', $data->trx);
        return to_route('deposit.confirm');
    }

    public function appDepositConfirm($hash) {
        try {
            $id = decrypt($hash);
        } catch (\Exception $ex) {
            abort(404);
        }
        $data = Deposit::where('id', $id)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->firstOrFail();
        $user = User::findOrFail($data->user_id);
        auth()->login($user);
        session()->put('Track', $data->trx);
        return to_route('deposit.confirm');
    }

    public function depositConfirm() {
        $track   = session()->get('Track');
        $deposit = Deposit::where('trx', $track)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->with('gateway')->firstOrFail();

        if ($deposit->method_code >= 1000) {
            return to_route('deposit.manual.confirm');
        }

        $dirName = $deposit->gateway->alias;
        $new     = __NAMESPACE__ . '\\' . $dirName . '\\ProcessController';

        $data = $new::process($deposit);
        $data = json_decode($data);

        if (isset($data->error)) {
            $notify[] = ['error', $data->message];
            return back()->withNotify($notify);
        }
        if (isset($data->redirect)) {
            return redirect($data->redirect_url);
        }

        // for Stripe V3
        if (@$data->session) {
            $deposit->btc_wallet = $data->session->id;
            $deposit->save();
        }
        // dd($data->view);
        $pageTitle = 'Payment Confirm';
        return view("Template::$data->view", compact('data', 'pageTitle', 'deposit'));
    }

    public static function userDataUpdate($deposit, $isManual = null) {
        if ($deposit->status == Status::PAYMENT_INITIATE || $deposit->status == Status::PAYMENT_PENDING) {
            $deposit->status = Status::PAYMENT_SUCCESS;
            $deposit->save();

            $donation         = $deposit->donation;
            $donation->status = Status::DONATION_PAID;
            $donation->save();

            $campaign = $donation->campaign;

            $user = $campaign->user;
            $user->balance += $deposit->amount;
            $user->save();

            $methodName = $deposit->methodName();

            $transaction               = new Transaction();
            $transaction->user_id      = $deposit->user_id;
            $transaction->amount       = $deposit->amount;
            $transaction->post_balance = $user->balance;
            $transaction->charge       = $deposit->charge;
            $transaction->trx_type     = '+';
            $transaction->details      = "New donation: " . strLimit($campaign->title, 15);
            $transaction->trx          = $deposit->trx;
            $transaction->remark       = 'donation';
            $transaction->save();

            $raisedPercent = gs('raised_charge');
            $raisedCharge  = ($deposit->amount * $raisedPercent) / 100;

            $user->balance -= $raisedCharge;
            $user->save();

            //Raised - Charge //
            $transaction               = new Transaction();
            $transaction->user_id      = $user->id;
            $transaction->amount       = $raisedCharge;
            $transaction->post_balance = $user->balance;
            $transaction->charge       = 0;
            $transaction->trx_type     = '-';
            $transaction->details      = "Raise charge for: " . strLimit($campaign->title, 15);
            $transaction->trx          = $deposit->trx;
            $transaction->remark       = 'raised_charge';
            $transaction->save();

            if (!$isManual) {
                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Deposit successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();
            }

            if ($deposit?->donation?->anonymous) {
                $donor = "Anonymous Donation";
            } else {
                $donor = auth()->check() ? auth()->user()->fullname : "Anonymous Donation";
            }

            //Referral Commission
            if (gs('donate_commission') && @$deposit->donation->user_id) {
                levelCommission($deposit->donation->user, $deposit->amount, 'donate_commission', $deposit->trx);
            }

            if (!$deposit?->donation?->anonymous) {
                notify(auth()->user(), $isManual ? 'DEPOSIT_APPROVE' : 'DEPOSIT_COMPLETE', [
                    'method_name'     => $methodName,
                    'method_currency' => $deposit->method_currency,
                    'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                    'amount'          => showAmount($deposit->amount, currencyFormat: false),
                    'charge'          => showAmount($deposit->charge, currencyFormat: false),
                    'rate'            => showAmount($deposit->rate, currencyFormat: false),
                    'trx'             => $deposit->trx,
                    'donor'           => $donor,
                    'post_balance'    => showAmount($user->balance, currencyFormat: false),
                ]);
            }
            notify($user, 'DONATION_RAISED', [
                'raised_amount' => showAmount($donation->donation),
                'donor_name'    => $donation->anonymous ? 'Anonymous' : auth()->user()?->fullname,
                'raised_charge' => showAmount($raisedCharge, currencyFormat: false),
                'raised_date'   => showDateTime($donation->created_at),
                'post_balance'  => showAmount($user->balance, currencyFormat: false),
            ]);
        }
    }

    public function manualDepositConfirm() {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        if ($data->method_code > 999) {
            $pageTitle = 'Confirm Payment';
            $method    = $data->gatewayCurrency();
            $gateway   = $method->method;
            return view('Template::user.payment.manual', compact('data', 'pageTitle', 'method', 'gateway'));
        }
        abort(404);
    }

    public function manualDepositUpdate(Request $request) {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        $gatewayCurrency = $data->gatewayCurrency();
        $gateway         = $gatewayCurrency->method;
        $formData        = $gateway->form->form_data;

        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $userData = $formProcessor->processFormData($request, $formData);

        $data->detail = $userData;
        $data->status = Status::PAYMENT_PENDING;
        $data->save();

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $data->user?->id ?? 0;
        $adminNotification->title     = 'Payment request to ' . $data->user?->username ?? "Anonymous";
        $adminNotification->click_url = urlPath('admin.deposit.details', $data->id);
        $adminNotification->save();

        notify($data->user, 'DEPOSIT_REQUEST', [
            'method_name'     => $data->gatewayCurrency()->name,
            'method_currency' => $data->method_currency,
            'method_amount'   => showAmount($data->final_amount, currencyFormat: false),
            'amount'          => showAmount($data->amount, currencyFormat: false),
            'charge'          => showAmount($data->charge, currencyFormat: false),
            'rate'            => showAmount($data->rate, currencyFormat: false),
            'trx'             => $data->trx,
        ]);

        $campaign = $data->donation->campaign;
        session()->put('THANK_YOU', [
            'campaign'      => @$campaign->title,
            'campaign_slug' => @$campaign->slug,
        ]);

        $notify[] = ['success', 'Your donation request has been successfully received.'];
        return to_route('campaign.index')->withNotify($notify);
    }
}
