<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Campaign;
use App\Models\CampaignUpdate;
use Carbon\Carbon;
use Exception;

class CampaignController extends Controller
{
    public function create()
    {
        $pageTitle = 'Start a Campaign';

        $categories = Category::active()->orderBy('id', 'DESC')->get();
        return view('Template::user.campaign.form', compact('pageTitle', 'categories'));
    }

    public function storeCampaign(Request $request, $id = 0)
    {
        $this->validation($request, $id);
        $category = Category::active()->find($request->category_id);

        if (!$category) {
            $notify[] = ['error', 'Invalid campaign category!'];
            return back()->withNotify($notify);
        }

        if ($id) {
            $campaign = Campaign::where('user_id', auth()->id())->findOrFail($id);
            $campaign->donor_visibility  = $request->donor_visibility == "on" ? 1 : 0;
            $notification = 'Campaign updated successfully, wait for admin approval';
        } else {
            $campaign = new Campaign();
            $notification = 'Campaign created successfully';
        }

        if ($request->hasFile('image')) {

            try {
                $path = getFilePath('campaign');
                $oldImage = '';
                $imageFilename = fileUploader($request->image, $path, getFileSize('campaign'), $oldImage);
                if ($id) {
                    $deleteImage = $path . '/' . $campaign->image;
                    fileManager()->removeFile($deleteImage);
                }
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload campaign image.'];
                return back()->withNotify($notify);
            }
        }
        if ($request->hasFile('attachments')) {
            try {
                $path = getFilePath('proof');
                $size = getFileSize('proof');
                $proofFiles = [];
                $key = $id ? count((array) $campaign->proof_images) : 0;
                $existingProofImages = $id ? (array) $campaign->proof_images : [];
                $pdfCount = count(array_filter($existingProofImages, fn($proof) => pathinfo($proof, PATHINFO_EXTENSION) == 'pdf'));
                $imageCount = count(array_filter($existingProofImages, fn($proof) => pathinfo($proof, PATHINFO_EXTENSION) != 'pdf'));

                foreach ($request->attachments as $attachment) {
                    $extension = $attachment->getClientOriginalExtension();
                    if ($extension == 'pdf') {
                        $filename = uniqid() . time() . '.' . $extension;
                        $proof = [
                            "proof_$key" => $filename,
                        ];
                        $proofFiles = $proofFiles + $proof;
                        $attachment->move($path, $filename);
                        $pdfCount++;
                    }
                    else {
                        $uploadedImage = fileUploader($attachment, $path, $size);
                        $image = [
                            "proof_$key" => $uploadedImage
                        ];
                        $proofFiles = $proofFiles + $image;
                        $imageCount++;
                    }
                    $key++;
                }
                $campaign->proof_images = array_merge($existingProofImages, $proofFiles);
                $campaign->save();
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Could not upload proof attachment!'];
                return back()->withNotify($notify);
            }
        }
        $filename  = @$filename ? $filename : $campaign->image;

        if ($id) {
            $proofData = array_merge($proofFiles ?? [], (array) $campaign->proof_images);
        } else {
            $proofData = $proofFiles;
        }

        $purifier  = new \HTMLPurifier();

        $campaign->category_id  = $request->category_id;

        $campaign->user_id      = auth()->user()->id;
        $campaign->title        = $request->title;
        $campaign->description  = $purifier->purify($request->description);
        $campaign->goal         = $request->goal;
        $campaign->goal_type    = $request->goal_type;
        $campaign->deadline     = $campaign->goal_type == Status::AFTER_DEADLINE  ? Carbon::parse($request->deadline) : NULL;
        $campaign->image        = $imageFilename;
        $campaign->proof_images = $proofData;
        $campaign->slug         = slug($request->title);
        $campaign->faqs = [
            'question' => $request->question,
            'answer' => $request->answer,
        ];


        $campaign->save();
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    protected function validation($request, $id)
    {
        $image          = $id ? 'nullable' : 'required';
        $proofDocuments = $id ? 'nullable' : 'required';
        $deadlineRule = $request->goal_type == Status::AFTER_DEADLINE ? 'required|' : 'nullable|';
        $request->validate([
            'category_id'   => 'required|integer',
            'title'         => "required|max:255|unique:campaigns,title,$id,id",
            'description'   => 'required|string',
            'goal_type'     => 'required|numeric|in:1,2,3',
            'goal'          => 'required|numeric|gt:0',
            'deadline'      => $deadlineRule . 'date|after:today',
            'image'         => [$image, new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'attachments'   => 'array|max:5',
            'attachments.*' => [$proofDocuments, new FileTypeValidate(['jpg', 'jpeg', 'png', 'pdf'])],
            'question.*'    => 'required|string',
            'answer.*'      => 'required|string'
        ]);
    }

    public function extended(Request $request, $id = 0)
    {
        $campaign = Campaign::where('user_id', auth()->id())->where('id', $id)->expired()->first();

        if (!$campaign) {
            $notify[] = ['error', 'The request to extend the campaign is invalid'];
            return back()->withNotify($notify);
        }

        $request->validate([
            'goal'          => 'required|numeric|gte:0',
            'deadline'      => 'required|date|after:today',
            'final_goal'    => 'required|numeric|gt:0',
        ]);

        $campaign->deadline    = Carbon::parse($request->deadline);
        $campaign->goal        = $request->goal + $campaign->goal;
        $campaign->extend_goal = $request->goal;
        $campaign->is_extend   = Status::YES;
        $campaign->expired     = Status::NO;
        $campaign->status      = Status::PENDING;
        $campaign->save();
        $notify[] = ['success', 'The campaign extension request has been successfully submitted to the author.'];
        return back()->withNotify($notify);
    }

    public function edit($id)
    {
        $pageTitle  = "Edit Campaign ";
        $campaign   = Campaign::where('user_id', auth()->id())->findOrFail($id);
        $categories = Category::active()->get();
        return view('Template::user.campaign.edit', compact('pageTitle', 'campaign', 'categories'));
    }

    protected function campaignData($scope = null)
    {
        if ($scope) {
            $campaigns = Campaign::$scope();
        } else {
            $campaigns = Campaign::query();
        }
        return $campaigns->where('user_id', auth()->id())->searchable(['title'])->with('donations', 'category')->orderBy('id', 'DESC')->paginate(getPaginate());
    }

    public function approvedCampaign()
    {
        $pageTitle  = "Approved Campaigns";
        $campaigns  = $this->campaignData('running');
        return view('Template::user.campaign.index', compact('campaigns', 'pageTitle'));
    }

    public function pendingCampaign()
    {
        $pageTitle  = "Pending Campaigns";
        $campaigns = $this->campaignData('pending');
        return view('Template::user.campaign.index', compact('campaigns', 'pageTitle'));
    }

    public function rejectedCampaign()
    {
        $pageTitle  = "Rejected Campaigns";
        $campaigns = $this->campaignData('rejected');
        return view('Template::user.campaign.index', compact('campaigns', 'pageTitle'));
    }

    public function completeCampaign()
    {
        $pageTitle = "Successful Campaigns";
        $campaigns = $this->campaignData('success');
        return view('Template::user.campaign.index', compact('campaigns', 'pageTitle'));
    }

    public function expiredCampaign()
    {
        $pageTitle = "Expired Campaigns";
        $campaigns = $this->campaignData('expired');
        return view('Template::user.campaign.index', compact('campaigns', 'pageTitle'));
    }

    public function runAndStop($id)
    {
        $campaign = Campaign::where('user_id', auth()->id())->findOrFail($id);
        if ($campaign->stop) {
            $campaign->stop = Status::NO;
            $notification = 'Campaign started successfully';
        } else {
            $campaign->stop = Status::YES;
            $notification = 'Campaign stopped successfully';
        }
        $campaign->save();
        $notify[] = ['success', $notification];
        return back()->withNotify($notify);
    }

    public function complete($id)
    {
        $campaign = Campaign::active()->findOrFail($id);
        $campaign->completed = Status::YES;
        $campaign->save();
        $notify[] = ['success', 'Campaign Completed Successfully'];
        return back()->withNotify($notify);
    }

    public function delete($id)
    {
        $campaign = Campaign::findOrFail($id);
        try {
            $path = getFilePath('campaign') . '/' . $campaign->image;
            if (!empty($campaign->proof_images)) {
                foreach ($campaign->proof_images as  $proof) {
                    $proofPath = getFilePath('proof') . '/' . $proof;
                    fileManager()->removeFile($proofPath);
                }
            }
            fileManager()->removeFile($path);
            $campaign->delete();
        } catch (Exception $ex) {
            $notify[] = ['error', $ex->getMessage()];
            return back()->withNotify($notify);
        }
        $notify[] = ['success', 'Campaign deleted Successfully'];
        return back()->withNotify($notify);
    }

    public function allCampaign()
    {
        $pageTitle = "All Campaigns";
        $campaigns = Campaign::searchable(['title'])->where('user_id', auth()->user()->id)->with('donations', 'category')->orderBy('id', "desc")->paginate(getPaginate());
        return view('Template::user.campaign.all', compact('pageTitle', 'campaigns'));
    }

    public function campaignUpdation($id)
    {
        $campaign = Campaign::where('user_id', auth()->id())->findOrFail($id);
        $pageTitle = $campaign->title;
        return view('Template::user.campaign.update', compact('pageTitle', 'campaign'));
    }

    public function campaignUpdationStore(Request $request, $id)
    {
        $request->validate([
            'updation' => 'required|string'
        ]);
        $campaign = Campaign::where('user_id', auth()->id())->findOrFail($id);

        $updation = CampaignUpdate::where('campaign_id', $campaign->id)->first();
        if (!$updation) {
            $updation = new CampaignUpdate;
        }
        $updation->campaign_id = $campaign->id;
        $updation->updation = $request->updation;
        $updation->save();
        $notify[] = ['success', 'Campaign updation successfully!'];
        return back()->withNotify($notify);
    }


    public function frontendSeo($id)
    {
        $data = Campaign::findOrFail($id);
        $pageTitle = 'SEO Configuration: ' . $data->title;
        return view('Template::user.campaign.frontend_seo', compact('pageTitle', 'data'));
    }

    public function frontendSeoUpdate(Request $request, $id)
    {
        $request->validate([
            'social_title'       => 'nullable|string',
            'social_description' => 'nullable|string',
            'keywords'           => 'nullable|array',
            'keywords.*'         => 'string',
            'description'        => 'nullable|string',
            'image'              => ['nullable', new FileTypeValidate(['jpeg', 'jpg', 'png'])]
        ]);


        $data = Campaign::findOrFail($id);
        $image = @$data->seo_content->image;
        if ($request->hasFile('image')) {
            try {
                $path = 'assets/images/frontend/campaign/seo';
                $image = fileUploader($request->image, $path, getFileSize('seo'), @$data->seo_content->image);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload the image'];
                return back()->withNotify($notify);
            }
        }

        $data->seo_content = [
            'image'              => $image,
            'description'        => $request->description,
            'social_title'       => $request->social_title,
            'social_description' => $request->social_description,
            'keywords'           => $request->keywords,
        ];

        $data->save();

        $notify[] = ['success', 'SEO content updated successfully'];
        return back()->withNotify($notify);
    }


    public function documentRemove($id, $key)
    {
        $campaign = Campaign::where('user_id', auth()->id())->findOrFail($id);
        $proofImages = (array) $campaign->proof_images;

        if (!array_key_exists($key, $proofImages)) {
            $notify[] = ['success', 'Document not found'];
            return back()->withNotify($notify);
        }

        $fileName = $proofImages[$key];
        $filePath = getFilePath('proof') . '/' . $fileName;

        if (file_exists($filePath)) {
            fileManager()->removeFile($filePath);
        }

        unset($proofImages[$key]);

        $campaign->proof_images = (object) $proofImages;
        $campaign->save();


        $notify[] = ['success', 'Document removed successfully'];
        return back()->withNotify($notify);
    }
}
